# 画面設計書 4-Base64 Encoding

## 概要

本ドキュメントは、VBCorLibプロジェクトのサンプルアプリケーション「Base64 Encoding」の画面設計書です。

### 本画面の処理概要

Base64 Encodingは、VBCorLibライブラリが提供するConvertクラスを使用して、ファイルのBase64エンコード/デコード機能をデモンストレーションする画面です。選択したファイルをBase64文字列に変換して表示し、その文字列をファイルにデコードして出力することができます。

**業務上の目的・背景**：Base64エンコードは、バイナリデータをテキスト形式で扱うための標準的な手法であり、メール添付やWebAPI通信などで広く使用されています。VBCorLibのConvertクラスは.NET FrameworkのConvertクラスと同様のAPIを提供し、VB6/VBAでも同等のBase64処理が可能です。本画面は、ファイルのBase64変換機能の使用方法と処理性能を開発者に示すことを目的としています。

**画面へのアクセス方法**：Examples/Misc/Base64 Encoding/Base64 Encoding.vbpプロジェクトを開き、F5キーで実行することでこの画面が表示されます。

**主要な操作・処理内容**：
1. 「Browse」ボタンで入力ファイルを選択
2. 「Encode Base64」ボタンでファイルをBase64エンコードし結果を表示
3. 「Browse」ボタン（New File）で出力先ファイルを指定
4. 「Decode To New File」ボタンでBase64文字列をデコードしてファイル出力

**画面遷移**：本画面は独立したデモアプリケーションであり、他の画面への遷移はありません。

**権限による表示制御**：本画面は開発者向けデモツールであり、特別な権限制御は実装されていません。

## 関連機能

| 機能No | 機能名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 13 | Convert | 主機能 | Base64エンコード/デコード機能（ToBase64String、FromBase64String） |
| 34 | File | 主機能 | ファイル操作（ReadAllBytes、WriteAllBytes、Exists、Delete） |

## 画面種別

ファイル変換 / デモンストレーション

## URL/ルーティング

VB6デスクトップアプリケーションのため、URLルーティングは存在しません。フォームファイル: `Examples/Misc/Base64 Encoding/Form1.frm`

## 入出力項目

| 項目名 | コントロール | 項目種別 | データ型 | 説明 |
|--------|-------------|----------|----------|------|
| Text1 | TextBox | 入力 | String | 入力ファイルパス |
| Text2 | TextBox | 入力 | String | 出力ファイルパス |
| RichTextBox1 | RichTextBox | 入出力 | String | エンコード結果表示/デコード元 |

## 表示項目

| 項目名 | コントロール | データ型 | 説明 |
|--------|-------------|----------|------|
| RichTextBox1 | RichTextBox | String | Base64エンコード結果（ReadOnly=True） |
| lblFileLength | Label | String | 入力ファイルサイズ |
| lblEncodedLength | Label | String | エンコード後の文字列長 |
| lblTime | Label | String | 処理時間（ミリ秒） |

## イベント仕様

### 1-cmdBrowse_Click

入力ファイルを選択するダイアログを表示します。

**処理内容**:
1. CommonDialogでファイル選択ダイアログ表示
2. 選択されたファイルパスをText1に設定

**コード参照**: 行170-178

```vb
Private Sub cmdBrowse_Click()
    On Error GoTo errTrap
    With CD
        .CancelError = True
        .ShowOpen
        Text1.Text = .FileName
    End With
errTrap:
End Sub
```

### 2-cmdBrowseNewFile_Click

出力ファイルを選択するダイアログを表示します。

**処理内容**:
1. CommonDialogでファイル選択ダイアログ表示
2. 選択されたファイルパスをText2に設定

**コード参照**: 行181-189

### 3-cmdEncode_Click

ファイルをBase64エンコードして表示します。

**処理内容**:
1. File.Existsで入力ファイルの存在確認
2. StopWatch.StartNewで計測開始
3. File.ReadAllBytesでファイル読み込み
4. Convert.ToBase64StringでBase64エンコード
5. 処理時間、ファイル長、エンコード長を表示
6. RichTextBox1にエンコード結果を表示

**コード参照**: 行220-247

```vb
Private Sub cmdEncode_Click()
    If Not File.Exists(Text1.Text) Then
        MsgBox "File does not exist.", vbExclamation + vbOKOnly, "File Not Found."
        Exit Sub
    End If

    Dim sw As StopWatch
    Set sw = StopWatch.StartNew

    Dim b() As Byte
    Dim s As String

    b = File.ReadAllBytes(Text1.Text)
    s = Convert.ToBase64String(b)

    lblTime.Caption = sw.ElapsedMilliseconds
    lblFileLength = FileLen(Text1.Text)
    lblEncodedLength.Caption = Len(s)
    RichTextBox1.Text = s
End Sub
```

### 4-cmdDecode_Click

Base64文字列をデコードしてファイルに出力します。

**処理内容**:
1. File.Existsで出力先ファイルの存在確認（上書き確認）
2. File.Deleteで既存ファイル削除（必要時）
3. RichTextBoxからBase64文字列取得
4. StopWatch.StartNewで計測開始
5. Convert.FromBase64Stringでデコード
6. 処理時間を表示
7. File.WriteAllBytesでファイル出力

**コード参照**: 行192-217

```vb
Private Sub cmdDecode_Click()
    If File.Exists(Text2.Text) Then
        If MsgBox("File already exists, Overwrite?", vbYesNo) = vbNo Then Exit Sub
        File.Delete Text2.Text
    End If

    Dim s As String
    s = RichTextBox1.Text

    Dim sw As StopWatch
    Set sw = StopWatch.StartNew

    Dim b() As Byte
    b = Convert.FromBase64String(s)

    lblTime.Caption = sw.ElapsedMilliseconds

    File.WriteAllbytes Text2.Text, b
End Sub
```

## データベース更新仕様

本画面はデータベースを使用しません。

## メッセージ仕様

| メッセージID | メッセージ内容 | 表示条件 |
|-------------|---------------|----------|
| MSG-01 | "File does not exist." | 入力ファイルが存在しない場合 |
| MSG-02 | "File already exists, Overwrite?" | 出力先ファイルが既に存在する場合 |

## 例外処理

| 例外種別 | 対処方法 |
|---------|---------|
| ファイル選択キャンセル | On Error GoTo errTrapで処理続行 |
| ファイル不存在 | MsgBox表示後、処理中止 |
| ファイル書き込みエラー | VBランタイムエラー発生 |

## 備考

- COMDLG32.OCX（CommonDialog）を使用
- RICHTX32.OCX（RichTextBox）を使用
- RichTextBoxはReadOnly=True、Courier 9.75ptフォント
- MIME 76文字改行ルールでBase64エンコード（コメントより）
- StopWatchクラスで処理時間を計測

---

## コードリーディングガイド

本画面を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Base64エンコード/デコードで使用されるデータ型を理解します。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Form1.frm | `Examples/Misc/Base64 Encoding/Form1.frm` | 行162-166: コメント説明 |

**読解のコツ**: VB6のByte配列とStringの変換がBase64処理の核心です。File.ReadAllBytesでByte()を取得し、Convert.ToBase64StringでStringに変換します。

#### Step 2: エントリーポイントを理解する

ユーザー操作（ボタンクリック）がエントリーポイントです。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Form1.frm | `Examples/Misc/Base64 Encoding/Form1.frm` | 行220-247: cmdEncode_Click |
| 2-2 | Form1.frm | `Examples/Misc/Base64 Encoding/Form1.frm` | 行192-217: cmdDecode_Click |

**主要処理フロー（Encode）**:
1. **行223-226**: ファイル存在確認
2. **行229-230**: StopWatch開始
3. **行235**: File.ReadAllBytesでバイト配列取得
4. **行236**: Convert.ToBase64Stringでエンコード
5. **行239-246**: 結果表示

**主要処理フロー（Decode）**:
1. **行194-197**: 出力先ファイル存在確認・上書き確認
2. **行203**: RichTextBoxからBase64文字列取得
3. **行211**: Convert.FromBase64Stringでデコード
4. **行216**: File.WriteAllBytesでファイル出力

#### Step 3: 補助機能を理解する

ファイル選択ダイアログの処理を確認します。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Form1.frm | `Examples/Misc/Base64 Encoding/Form1.frm` | 行170-178: cmdBrowse_Click |
| 3-2 | Form1.frm | `Examples/Misc/Base64 Encoding/Form1.frm` | 行181-189: cmdBrowseNewFile_Click |

### プログラム呼び出し階層図

```
cmdBrowse_Click
    │
    └─ CommonDialog.ShowOpen
           │
           └─ Text1.Text = FileName

cmdEncode_Click
    │
    ├─ File.Exists (チェック)
    │
    ├─ StopWatch.StartNew
    │
    ├─ File.ReadAllBytes
    │      │
    │      └─ Byte配列取得
    │
    ├─ Convert.ToBase64String
    │      │
    │      └─ Base64文字列生成
    │
    └─ 結果表示
           ├─ lblTime
           ├─ lblFileLength
           ├─ lblEncodedLength
           └─ RichTextBox1.Text

cmdDecode_Click
    │
    ├─ File.Exists (上書き確認)
    │      │
    │      └─ File.Delete (必要時)
    │
    ├─ StopWatch.StartNew
    │
    ├─ Convert.FromBase64String
    │      │
    │      └─ Byte配列復元
    │
    └─ File.WriteAllBytes
           │
           └─ ファイル出力
```

### データフロー図

```
[入力]               [処理]                    [出力]

入力ファイル ────▶ File.ReadAllBytes ───▶ Byte配列
                          │
                          ▼
               Convert.ToBase64String
                          │
                          ▼
                    Base64文字列 ────────▶ RichTextBox1

RichTextBox1 ────▶ Convert.FromBase64String
                          │
                          ▼
                     Byte配列
                          │
                          ▼
               File.WriteAllBytes ────────▶ 出力ファイル
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Form1.frm | `Examples/Misc/Base64 Encoding/Form1.frm` | ソース | メインフォーム |
| Form1.frx | `Examples/Misc/Base64 Encoding/Form1.frx` | バイナリ | フォームリソース |
| Base64 Encoding.vbp | `Examples/Misc/Base64 Encoding/Base64 Encoding.vbp` | プロジェクト | プロジェクト定義 |
| COMDLG32.OCX | 外部 | OCX | CommonDialogコントロール |
| RICHTX32.OCX | 外部 | OCX | RichTextBoxコントロール |
| VBCorLib.dll | 外部 | DLL | VBCorLibライブラリ本体 |
